/*
 * Copyright (c) 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.jpackage.internal;

import static jdk.jpackage.internal.ApplicationImageUtils.createLauncherIconResource;
import jdk.jpackage.internal.PackagingPipeline.AppImageBuildEnv;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import jdk.jpackage.internal.PackagingPipeline.BuildApplicationTaskID;
import jdk.jpackage.internal.PackagingPipeline.PrimaryTaskID;
import jdk.jpackage.internal.PackagingPipeline.TaskID;
import jdk.jpackage.internal.model.Application;
import jdk.jpackage.internal.model.ApplicationLayout;
import jdk.jpackage.internal.resources.ResourceLocator;

final class LinuxPackagingPipeline {

    enum LinuxAppImageTaskID implements TaskID {
        LAUNCHER_LIB,
        LAUNCHER_ICONS
    }

    static PackagingPipeline.Builder build() {
        return PackagingPipeline.buildStandard()
                .task(LinuxAppImageTaskID.LAUNCHER_LIB)
                        .addDependent(PrimaryTaskID.BUILD_APPLICATION_IMAGE)
                        .applicationAction(LinuxPackagingPipeline::writeLauncherLib).add()
                .task(LinuxAppImageTaskID.LAUNCHER_ICONS)
                        .addDependent(BuildApplicationTaskID.CONTENT)
                        .applicationAction(LinuxPackagingPipeline::writeLauncherIcons).add();
    }

    private static void writeLauncherLib(
            AppImageBuildEnv<Application, LinuxApplicationLayout> env) throws IOException {

        final var launcherLib = env.resolvedLayout().libAppLauncher();
        try (var in = ResourceLocator.class.getResourceAsStream("libjpackageapplauncheraux.so")) {
            Files.createDirectories(launcherLib.getParent());
            Files.copy(in, launcherLib);
        }
    }

    private static void writeLauncherIcons(
            AppImageBuildEnv<Application, ApplicationLayout> env) throws IOException {

        for (var launcher : env.app().launchers()) {
            createLauncherIconResource(env.app(), launcher, env.env()::createResource).ifPresent(iconResource -> {
                String iconFileName = launcher.executableName() + ".png";
                Path iconTarget = env.resolvedLayout().desktopIntegrationDirectory().resolve(iconFileName);
                try {
                    iconResource.saveToFile(iconTarget);
                } catch (IOException ex) {
                    throw new UncheckedIOException(ex);
                }
            });
        }
    }

    static final LinuxApplicationLayout APPLICATION_LAYOUT = LinuxApplicationLayout.create(
            ApplicationLayoutUtils.PLATFORM_APPLICATION_LAYOUT, Path.of("lib/libapplauncher.so"));
}
