# frozen_string_literal: true
#
# Author:: Paul Rossman (<paulrossman@google.com>)
# Author:: Chef Partner Engineering (<partnereng@chef.io>)
# Copyright:: Copyright 2015-2016 Google Inc., Chef Software, Inc.
# License:: Apache License, Version 2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require "chef/knife"
require "chef/knife/cloud/list_resource_command"
require_relative "cloud/google_service_options"

class Chef::Knife::Cloud
  class GoogleZoneList < ResourceListCommand
    include GoogleServiceOptions

    banner "knife google zone list"

    deps do
      require_relative "cloud/google_service"
      require_relative "cloud/google_service_helpers"
      include GoogleServiceHelpers
    end

    def validate_params!
      check_for_missing_config_values!
      super
    end

    def before_exec_command
      @columns_with_info = [
        { label: "Zone",   key: "name" },
        { label: "Status", key: "status", value_callback: method(:format_status_value) },
      ]

      @sort_by_field = "name"
    end

    def format_status_value(status)
      status = status.downcase
      status_color = if status == "up"
                       :green
                     else
                       :red
                     end

      ui.color(status, status_color)
    end

    def query_resource
      service.list_zones
    end
  end
end
