# frozen_string_literal: true
#
# Copyright:: 2019, Chef Software, Inc.
# Author:: Tim Smith (<tsmith84@gmail.com>)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
module RuboCop
  module Cop
    module Chef
      module Modernize
        # ChefSpec 7.1 and later auto generate ChefSpec matchers. Matchers in cookbooks can now be removed.
        #
        # @example
        #
        #   ### incorrect
        #   if defined?(ChefSpec)
        #     def create_yum_repository(resource_name)
        #       ChefSpec::Matchers::ResourceMatcher.new(:yum_repository, :create, resource_name)
        #     end
        #   end
        #
        class DefinesChefSpecMatchers < Base
          extend AutoCorrector
          include RangeHelp

          MSG = 'ChefSpec matchers are now auto generated by ChefSpec 7.1+ and do not need to be defined in a cookbook'

          def_node_matcher :chefspec_matcher?, <<-PATTERN
            (if ( :defined? (const nil? :ChefSpec)) ... )
          PATTERN

          def on_if(node)
            chefspec_matcher?(node) do
              add_offense(node, severity: :refactor) do |corrector|
                corrector.remove(range_with_surrounding_space(range: node.loc.expression, side: :left))
              end
            end
          end
        end
      end
    end
  end
end
